// Hook Me Up - JavaScript Functionality

// Configuration
const CONFIG = {
    whatsappNumber: '+254758256440', // Variable for easy editing
    companyName: 'Hook Me Up',
    baseUrl: window.location.origin,
    apiEndpoint: '/api', // For future backend integration
};

// Global state
let userLocation = null;
let currentUser = null;
let services = [];

// DOM Content Loaded
document.addEventListener('DOMContentLoaded', function() {
    initializeApp();
});

// Initialize Application
function initializeApp() {
    // Set up event listeners
    setupEventListeners();
    
    // Initialize geolocation
    initializeGeolocation();
    
    // Setup smooth scrolling
    setupSmoothScrolling();
    
    // Initialize navbar scroll effect
    setupNavbarScroll();
    
    // Load services data
    loadServices();
    
    // Set up form validations
    setupFormValidations();
    
    // Initialize animations
    initializeAnimations();
    
    // Update contact numbers
    updateContactNumbers();
}

// Event Listeners Setup
function setupEventListeners() {
    // Navigation toggle for mobile
    const navToggle = document.getElementById('nav-toggle');
    const navMenu = document.getElementById('nav-menu');
    
    if (navToggle && navMenu) {
        navToggle.addEventListener('click', toggleMobileNav);
    }
    
    // Modal close on outside click
    document.addEventListener('click', handleModalOutsideClick);
    
    // Form submissions
    const contactForm = document.getElementById('contact-form');
    const loginForm = document.getElementById('login-form');
    const signupForm = document.getElementById('signup-form');
    
    if (contactForm) {
        contactForm.addEventListener('submit', handleContactForm);
    }
    
    if (loginForm) {
        loginForm.addEventListener('submit', handleLogin);
    }
    
    if (signupForm) {
        signupForm.addEventListener('submit', handleSignup);
    }
    
    // Service card clicks
    document.querySelectorAll('.service-card').forEach(card => {
        card.addEventListener('click', handleServiceCardClick);
    });
    
    // Keyboard navigation
    document.addEventListener('keydown', handleKeyboardNavigation);
}

// Geolocation Functions
function initializeGeolocation() {
    if (navigator.geolocation) {
        navigator.geolocation.getCurrentPosition(
            handleLocationSuccess,
            handleLocationError,
            {
                enableHighAccuracy: true,
                timeout: 10000,
                maximumAge: 300000 // 5 minutes
            }
        );
    } else {
        console.log('Geolocation is not supported by this browser.');
        showLocationPrompt();
    }
}

function handleLocationSuccess(position) {
    userLocation = {
        latitude: position.coords.latitude,
        longitude: position.coords.longitude,
        accuracy: position.coords.accuracy
    };
    
    console.log('Location obtained:', userLocation);
    
    // Update UI with location-based content
    updateLocationBasedContent();
    
    // Show location success message
    showNotification('Location detected! Finding companions near you.', 'success');
}

function handleLocationError(error) {
    let message = 'Unable to retrieve your location.';
    
    switch(error.code) {
        case error.PERMISSION_DENIED:
            message = 'Location access denied by user.';
            break;
        case error.POSITION_UNAVAILABLE:
            message = 'Location information is unavailable.';
            break;
        case error.TIMEOUT:
            message = 'Location request timed out.';
            break;
    }
    
    console.error('Location error:', message);
    showLocationPrompt();
}

function showLocationPrompt() {
    // Show a user-friendly prompt to enable location
    const prompt = document.createElement('div');
    prompt.className = 'location-prompt';
    prompt.innerHTML = `
        <div class="prompt-content">
            <i class="fas fa-map-marker-alt"></i>
            <h4>Enable Location Access</h4>
            <p>Allow location access to find companions near you in Nairobi</p>
            <button onclick="requestLocation()" class="btn-primary">Enable Location</button>
            <button onclick="dismissLocationPrompt()" class="btn-outline">Maybe Later</button>
        </div>
    `;
    document.body.appendChild(prompt);
}

function requestLocation() {
    initializeGeolocation();
    dismissLocationPrompt();
}

function dismissLocationPrompt() {
    const prompt = document.querySelector('.location-prompt');
    if (prompt) {
        prompt.remove();
    }
}

function updateLocationBasedContent() {
    if (!userLocation) return;
    
    // Update location-specific content
    const locationElements = document.querySelectorAll('[data-location]');
    locationElements.forEach(element => {
        // Update with actual location-based data
        element.textContent = 'Near you in Nairobi';
    });
}

// Navigation Functions
function toggleMobileNav() {
    const navMenu = document.getElementById('nav-menu');
    const navToggle = document.getElementById('nav-toggle');
    
    navMenu.classList.toggle('active');
    navToggle.classList.toggle('active');
}

function setupSmoothScrolling() {
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
        anchor.addEventListener('click', function (e) {
            e.preventDefault();
            const target = document.querySelector(this.getAttribute('href'));
            if (target) {
                target.scrollIntoView({
                    behavior: 'smooth',
                    block: 'start'
                });
                
                // Close mobile nav if open
                const navMenu = document.getElementById('nav-menu');
                const navToggle = document.getElementById('nav-toggle');
                if (navMenu.classList.contains('active')) {
                    navMenu.classList.remove('active');
                    navToggle.classList.remove('active');
                }
            }
        });
    });
}

function setupNavbarScroll() {
    const navbar = document.getElementById('navbar');
    let lastScrollTop = 0;
    
    window.addEventListener('scroll', function() {
        const scrollTop = window.pageYOffset || document.documentElement.scrollTop;
        
        // Add scrolled class for styling
        if (scrollTop > 100) {
            navbar.classList.add('scrolled');
        } else {
            navbar.classList.remove('scrolled');
        }
        
        // Hide/show navbar on scroll (optional)
        if (scrollTop > lastScrollTop && scrollTop > 200) {
            navbar.style.transform = 'translateY(-100%)';
        } else {
            navbar.style.transform = 'translateY(0)';
        }
        
        lastScrollTop = scrollTop <= 0 ? 0 : scrollTop;
    });
}

// Service Functions
function loadServices() {
    // Mock services data - in production, this would come from API
    services = [
        {
            id: 'events',
            name: 'Event Companions',
            description: 'Professional companions for weddings, parties, corporate events',
            price: 'From KSh 2,000/event',
            icon: 'fas fa-calendar-alt',
            features: ['Wedding ceremonies', 'Corporate functions', 'Birthday parties', 'Cultural events']
        },
        {
            id: 'travel',
            name: 'Travel & Adventure Partners',
            description: 'Safe, verified travel companions for exploring Kenya and beyond',
            price: 'From KSh 1,500/day',
            icon: 'fas fa-plane',
            features: ['Local area tours', 'Weekend getaways', 'Adventure activities', 'Cultural experiences']
        },
        // Add more services as needed
    ];
}

function selectService(serviceId) {
    const service = services.find(s => s.id === serviceId);
    if (service) {
        // Open WhatsApp with pre-filled message
        const message = `Hi! I'm interested in ${service.name} service. Can you provide more information?`;
        contactWhatsApp(message);
    }
}

function handleServiceCardClick(event) {
    const card = event.currentTarget;
    const category = card.getAttribute('data-category');
    if (category) {
        selectService(category);
    }
}

// Contact Functions
function updateContactNumbers() {
    // Update all contact numbers with the configurable number
    const phoneElements = document.querySelectorAll('#phone-number, #whatsapp-number');
    phoneElements.forEach(element => {
        element.textContent = CONFIG.whatsappNumber;
    });
}

function contactWhatsApp(customMessage = null) {
    const message = customMessage || 'Hi! I\'m interested in learning more about Hook Me Up services.';
    const encodedMessage = encodeURIComponent(message);
    const whatsappUrl = `https://wa.me/${CONFIG.whatsappNumber.replace(/[^\d]/g, '')}?text=${encodedMessage}`;
    
    window.open(whatsappUrl, '_blank');
    
    // Track interaction
    trackEvent('contact', 'whatsapp', 'hero_cta');
}

function makeCall() {
    window.location.href = `tel:${CONFIG.whatsappNumber}`;
    
    // Track interaction
    trackEvent('contact', 'phone', 'direct_call');
}

// Modal Functions
function openLoginModal() {
    showModal('login-modal');
}

function openSignupModal() {
    showModal('signup-modal');
}

function showModal(modalId) {
    const modal = document.getElementById(modalId);
    if (modal) {
        modal.classList.add('active');
        document.body.style.overflow = 'hidden';
        
        // Focus on first input
        const firstInput = modal.querySelector('input');
        if (firstInput) {
            setTimeout(() => firstInput.focus(), 100);
        }
    }
}

function closeModal(modalId) {
    const modal = document.getElementById(modalId);
    if (modal) {
        modal.classList.remove('active');
        document.body.style.overflow = '';
    }
}

function switchModal(fromModalId, toModalId) {
    closeModal(fromModalId);
    setTimeout(() => showModal(toModalId), 200);
}

function handleModalOutsideClick(event) {
    if (event.target.classList.contains('modal')) {
        event.target.classList.remove('active');
        document.body.style.overflow = '';
    }
}

// Form Handling
function setupFormValidations() {
    // Add real-time validation to forms
    const forms = document.querySelectorAll('form');
    forms.forEach(form => {
        const inputs = form.querySelectorAll('input, select, textarea');
        inputs.forEach(input => {
            input.addEventListener('blur', validateField);
            input.addEventListener('input', clearFieldError);
        });
    });
}

function validateField(event) {
    const field = event.target;
    const value = field.value.trim();
    let isValid = true;
    let errorMessage = '';
    
    // Remove existing error
    clearFieldError(event);
    
    // Required field validation
    if (field.hasAttribute('required') && !value) {
        isValid = false;
        errorMessage = 'This field is required.';
    }
    
    // Email validation
    if (field.type === 'email' && value) {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        if (!emailRegex.test(value)) {
            isValid = false;
            errorMessage = 'Please enter a valid email address.';
        }
    }
    
    // Phone validation (Kenyan format)
    if (field.type === 'tel' && value) {
        const phoneRegex = /^(\+254|0)[17]\d{8}$/;
        if (!phoneRegex.test(value)) {
            isValid = false;
            errorMessage = 'Please enter a valid Kenyan phone number.';
        }
    }
    
    // Password validation
    if (field.type === 'password' && value) {
        if (value.length < 8) {
            isValid = false;
            errorMessage = 'Password must be at least 8 characters long.';
        }
    }
    
    if (!isValid) {
        showFieldError(field, errorMessage);
    }
    
    return isValid;
}

function showFieldError(field, message) {
    field.classList.add('error');
    
    // Remove existing error message
    const existingError = field.parentNode.querySelector('.field-error');
    if (existingError) {
        existingError.remove();
    }
    
    // Add error message
    const errorElement = document.createElement('div');
    errorElement.className = 'field-error';
    errorElement.textContent = message;
    field.parentNode.appendChild(errorElement);
}

function clearFieldError(event) {
    const field = event.target;
    field.classList.remove('error');
    
    const errorElement = field.parentNode.querySelector('.field-error');
    if (errorElement) {
        errorElement.remove();
    }
}

function handleContactForm(event) {
    event.preventDefault();
    
    const form = event.target;
    const formData = new FormData(form);
    const data = Object.fromEntries(formData);
    
    // Validate form
    let isValid = true;
    const inputs = form.querySelectorAll('input[required], select[required], textarea[required]');
    inputs.forEach(input => {
        if (!validateField({ target: input })) {
            isValid = false;
        }
    });
    
    if (!isValid) {
        showNotification('Please correct the errors below.', 'error');
        return;
    }
    
    // Show loading state
    const submitBtn = form.querySelector('button[type="submit"]');
    const originalText = submitBtn.textContent;
    submitBtn.textContent = 'Sending...';
    submitBtn.disabled = true;
    
    // Simulate form submission - in production, send to backend
    setTimeout(() => {
        // Create WhatsApp message with form data
        const message = `New Contact Form Submission:
Name: ${data.name}
Email: ${data.email}
Phone: ${data.phone}
Service: ${data.service}
Message: ${data.message}`;
        
        // Send via WhatsApp
        contactWhatsApp(message);
        
        // Reset form
        form.reset();
        
        // Show success message
        showNotification('Message sent successfully! We\'ll get back to you soon.', 'success');
        
        // Reset button
        submitBtn.textContent = originalText;
        submitBtn.disabled = false;
        
        // Track form submission
        trackEvent('form', 'submit', 'contact_form');
    }, 1000);
}

function handleLogin(event) {
    event.preventDefault();
    
    const form = event.target;
    const formData = new FormData(form);
    const data = Object.fromEntries(formData);
    
    // Show loading state
    const submitBtn = form.querySelector('button[type="submit"]');
    const originalText = submitBtn.textContent;
    submitBtn.textContent = 'Signing In...';
    submitBtn.disabled = true;
    
    // Simulate login - in production, authenticate with backend
    setTimeout(() => {
        // Mock successful login
        currentUser = {
            email: data.email,
            name: 'John Doe', // Would come from backend
            memberSince: new Date(),
            verified: true
        };
        
        localStorage.setItem('hookmeup_user', JSON.stringify(currentUser));
        
        closeModal('login-modal');
        showNotification('Welcome back! You\'re now signed in.', 'success');
        updateUserInterface();
        
        // Reset button
        submitBtn.textContent = originalText;
        submitBtn.disabled = false;
        
        // Track login
        trackEvent('auth', 'login', 'modal');
    }, 1500);
}

function handleSignup(event) {
    event.preventDefault();
    
    const form = event.target;
    const formData = new FormData(form);
    const data = Object.fromEntries(formData);
    
    // Validate terms agreement
    if (!data['terms-agree']) {
        showNotification('Please agree to the Terms of Service and Privacy Policy.', 'error');
        return;
    }
    
    // Show loading state
    const submitBtn = form.querySelector('button[type="submit"]');
    const originalText = submitBtn.textContent;
    submitBtn.textContent = 'Creating Account...';
    submitBtn.disabled = true;
    
    // Simulate signup - in production, register with backend
    setTimeout(() => {
        // Mock successful signup
        currentUser = {
            email: data.email,
            name: data.name,
            phone: data.phone,
            memberSince: new Date(),
            verified: false
        };
        
        localStorage.setItem('hookmeup_user', JSON.stringify(currentUser));
        
        closeModal('signup-modal');
        showNotification('Account created successfully! Welcome to Hook Me Up.', 'success');
        updateUserInterface();
        
        // Reset button
        submitBtn.textContent = originalText;
        submitBtn.disabled = false;
        
        // Track signup
        trackEvent('auth', 'signup', 'modal');
    }, 1500);
}

// UI Update Functions
function updateUserInterface() {
    if (currentUser) {
        // Update navigation to show user menu
        updateNavigation();
        
        // Personalize content
        personalizeContent();
    }
}

function updateNavigation() {
    const navActions = document.querySelector('.nav-actions');
    if (navActions && currentUser) {
        navActions.innerHTML = `
            <div class="user-menu">
                <span class="user-greeting">Hi, ${currentUser.name.split(' ')[0]}!</span>
                <button class="btn-primary" onclick="openDashboard()">Dashboard</button>
                <button class="btn-outline" onclick="logout()">Logout</button>
            </div>
        `;
    }
}

function personalizeContent() {
    // Add personalized content based on user data
    const heroTitle = document.querySelector('.hero-title');
    if (heroTitle && currentUser) {
        heroTitle.innerHTML = `Welcome back, <span class="gradient-text">${currentUser.name.split(' ')[0]}</span>! Ready to connect?`;
    }
}

function logout() {
    currentUser = null;
    localStorage.removeItem('hookmeup_user');
    
    // Reset UI
    location.reload();
    
    showNotification('You\'ve been signed out successfully.', 'success');
    trackEvent('auth', 'logout', 'nav');
}

// Utility Functions
function getStarted() {
    if (currentUser) {
        openDashboard();
    } else {
        openSignupModal();
    }
    
    trackEvent('cta', 'click', 'hero_get_started');
}

function scrollToServices() {
    const servicesSection = document.getElementById('services');
    if (servicesSection) {
        servicesSection.scrollIntoView({ behavior: 'smooth' });
    }
    
    trackEvent('cta', 'click', 'hero_explore_services');
}

function openDashboard() {
    // In production, this would navigate to a dashboard page
    showNotification('Dashboard coming soon! Contact us via WhatsApp for now.', 'info');
    contactWhatsApp('Hi! I\'d like to access my dashboard and manage my account.');
}

function showNotification(message, type = 'info') {
    // Remove existing notifications
    const existingNotifications = document.querySelectorAll('.notification');
    existingNotifications.forEach(notification => notification.remove());
    
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <div class="notification-content">
            <i class="fas ${getNotificationIcon(type)}"></i>
            <span>${message}</span>
            <button class="notification-close" onclick="this.parentElement.parentElement.remove()">
                <i class="fas fa-times"></i>
            </button>
        </div>
    `;
    
    // Add to page
    document.body.appendChild(notification);
    
    // Auto remove after 5 seconds
    setTimeout(() => {
        if (notification.parentNode) {
            notification.remove();
        }
    }, 5000);
    
    // Animate in
    setTimeout(() => notification.classList.add('show'), 100);
}

function getNotificationIcon(type) {
    const icons = {
        success: 'fa-check-circle',
        error: 'fa-exclamation-circle',
        warning: 'fa-exclamation-triangle',
        info: 'fa-info-circle'
    };
    return icons[type] || icons.info;
}

// Animation Functions
function initializeAnimations() {
    // Intersection Observer for scroll animations
    const observerOptions = {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
    };
    
    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('fade-in');
                observer.unobserve(entry.target);
            }
        });
    }, observerOptions);
    
    // Observe elements for animation
    document.querySelectorAll('.service-card, .testimonial-card, .step, .pricing-card').forEach(el => {
        observer.observe(el);
    });
}

// Keyboard Navigation
function handleKeyboardNavigation(event) {
    // Close modal on Escape key
    if (event.key === 'Escape') {
        const activeModal = document.querySelector('.modal.active');
        if (activeModal) {
            activeModal.classList.remove('active');
            document.body.style.overflow = '';
        }
    }
    
    // Navigate with arrow keys (accessibility)
    if (event.key === 'Tab') {
        // Ensure proper tab order
        const focusableElements = document.querySelectorAll(
            'a[href], button, input, select, textarea, [tabindex]:not([tabindex="-1"])'
        );
        
        // Handle tab navigation within modals
        const activeModal = document.querySelector('.modal.active');
        if (activeModal) {
            const modalFocusableElements = activeModal.querySelectorAll(
                'a[href], button, input, select, textarea, [tabindex]:not([tabindex="-1"])'
            );
            
            if (modalFocusableElements.length > 0) {
                const firstElement = modalFocusableElements[0];
                const lastElement = modalFocusableElements[modalFocusableElements.length - 1];
                
                if (event.shiftKey && document.activeElement === firstElement) {
                    event.preventDefault();
                    lastElement.focus();
                } else if (!event.shiftKey && document.activeElement === lastElement) {
                    event.preventDefault();
                    firstElement.focus();
                }
            }
        }
    }
}

// Analytics & Tracking
function trackEvent(category, action, label = null) {
    // Google Analytics 4 event tracking
    if (typeof gtag !== 'undefined') {
        gtag('event', action, {
            event_category: category,
            event_label: label,
            custom_parameter: 'hook_me_up'
        });
    }
    
    // Console log for development
    console.log(`Event tracked: ${category} - ${action} - ${label}`);
}

// Error Handling
window.addEventListener('error', function(event) {
    console.error('JavaScript Error:', event.error);
    
    // Track errors (in production, send to error tracking service)
    trackEvent('error', 'javascript', event.error.message);
});

// Performance Monitoring
window.addEventListener('load', function() {
    // Track page load time
    const loadTime = performance.now();
    console.log(`Page loaded in ${loadTime.toFixed(2)}ms`);
    
    trackEvent('performance', 'page_load', Math.round(loadTime));
});

// Service Worker Registration (for future PWA features)
if ('serviceWorker' in navigator) {
    window.addEventListener('load', function() {
        navigator.serviceWorker.register('/sw.js')
            .then(function(registration) {
                console.log('ServiceWorker registration successful');
            })
            .catch(function(err) {
                console.log('ServiceWorker registration failed');
            });
    });
}

// Initialize user session
function initializeUserSession() {
    const savedUser = localStorage.getItem('hookmeup_user');
    if (savedUser) {
        try {
            currentUser = JSON.parse(savedUser);
            updateUserInterface();
        } catch (error) {
            console.error('Error parsing saved user data:', error);
            localStorage.removeItem('hookmeup_user');
        }
    }
}

// Call initialization functions
document.addEventListener('DOMContentLoaded', function() {
    initializeUserSession();
});

// Export functions for global access
window.HookMeUp = {
    contactWhatsApp,
    makeCall,
    openLoginModal,
    openSignupModal,
    closeModal,
    switchModal,
    selectService,
    getStarted,
    scrollToServices,
    showNotification,
    trackEvent
};